#include "RTL.h"                      /* RTL kernel functions & defines      */
#include <stdio.h>                    /* standard I/O .h-file                */
#include <ctype.h>                    /* character functions                 */
#include <string.h>                   /* string and memory functions         */
#include "File_Config.h"
#include "BL_SD.h"
#include "lpc2000_sector.h"
#include "IAP.h"


// for display of bootloader info
extern char Image$$ER_IROM1$$RO$$Base[];
extern char Image$$ER_IROM1$$RO$$Length[];
extern char Image$$RW_IRAM1$$RW$$Base[];
extern char Image$$RW_IRAM1$$RW$$Length[];

// for IAP
#define IAP_BUF_SIZE  1024
#define DEFAULT_RUN_ADDR 0x10000
#define DEFAULT_PROG_ADDR 0x10000
__align(4) U8  IAP_Buf[IAP_BUF_SIZE];

/* Pointer to Function returning Void (any number of parameters) */
typedef void (*PFV)();

// 	SWI handler
__swi(0x00) void SwiHandle1(int Handle);
#define IRQDisable() SwiHandle1(0)
#define IRQEnable()  SwiHandle1(1)

/* Command Functions */
static void cmd_dir (char *par);
static void cmd_help (char *par);
static void cmd_bootinfo (char *par);
static void cmd_prog (char *par);
static void cmd_run (char *par);
static void cmd_blkchk (char *par);
static void cmd_erase (char *par);
static void cmd_readid (char *par);

/* Local constants */
static const char intro[] =
   "\n\n"
   "+-----------------------------------------------------------------------+\n"
   "|                LPC2000 Bootloader via SD/MMC                          |\n";
static const char help[] = 
   "+ command ------------------+ function ---------------------------------+\n"
   "| INFO                      | display the bootloader info               |\n"
   "| READID                    | read part ID                              |\n"   
   "| DIR \"[mask]\"            | displays a list of files in SD/MMC        |\n"
   "| BLKCHK [from_idx] [to_idx]| check if flash sectors are blank          |\n"
   "| ERASE [from_idx] [to_idx] | erase flash sectors                       |\n"
   "| RUN [nnnn]                | Run application in internal FLASH         |\n"
   "|                           | [nnnn - app. address, default=0x10000     |\n"
   "| PROG \"fname.bin\" [nnnn] | Program flash using a BIN file            |\n"
   "|                           | [nnnn - app. address, default=0x10000     |\n"
   "| HELP  or  ?               | displays this help                        |\n"
   "+---------------------------+-------------------------------------------+\n";

static const SCMD cmd[] = {
   "INFO", 	 cmd_bootinfo,
   "BLKCHK", cmd_blkchk,
   "ERASE",  cmd_erase,
   "READID", cmd_readid,
   "RUN",    cmd_run,
   "PROG", 	 cmd_prog,
   "DIR",    cmd_dir,
   "HELP",   cmd_help,
   "?",      cmd_help };

#define CMD_COUNT   (sizeof (cmd) / sizeof (cmd[0]))

/* Local variables */
static char in_line[80];
static U32 bl_size, bl_sector_startidx, bl_sector_endidx;

/* Local Function Prototypes */
static char *get_entry (char *cp, char **pNext);
static void init_card (void);
static void init_bootpara(void);
static int strtoul(char *s);
static int IAP_Program(U32 sector_index, U32 app_addr);
static int IAP_PrepareErase(U32 sector_index);


/*----------------------------------------------------------------------------
 *        Process input string for long or short name entry
 *---------------------------------------------------------------------------*/
static char *get_entry (char *cp, char **pNext) {
   char *sp, lfn = 0, sep_ch = ' ';

   if (cp == NULL) {                          /* skip NULL pointers          */
      *pNext = cp;
      return (cp);
   }

   for ( ; *cp == ' ' || *cp == '\"'; cp++) { /* skip blanks and starting  " */
      if (*cp == '\"') { sep_ch = '\"'; lfn = 1; }
      *cp = 0;
   }
 
   for (sp = cp; *sp != CR && *sp != LF; sp++) {
      if ( lfn && *sp == '\"') break;
      if (!lfn && *sp == ' ' ) break;
   }

   for ( ; *sp == sep_ch || *sp == CR || *sp == LF; sp++) {
      *sp = 0;
      if ( lfn && *sp == sep_ch) { sp ++; break; }
   }

   *pNext = (*sp) ? sp : NULL;                /* next entry                  */
   return (cp);
}

/*----------------------------------------------------------------------------
 *        Print size in dotted fomat
 *---------------------------------------------------------------------------*/
static void dot_format (U32 val, char *sp) {

   if (val >= (U32)1e9) {
      sp += sprintf (sp,"%d.",val/(U32)1e9);
      val %= (U32)1e9;
      sp += sprintf (sp,"%03d.",val/(U32)1e6);
      val %= (U32)1e6;
      sprintf (sp,"%03d.%03d",val/1000,val%1000);
      return;
   }
   if (val >= (U32)1e6) {
      sp += sprintf (sp,"%d.",val/(U32)1e6);
      val %= (U32)1e6;
      sprintf (sp,"%03d.%03d",val/1000,val%1000);
      return;
   }
   if (val >= 1000) {
      sprintf (sp,"%d.%03d",val/1000,val%1000);
      return;
   }
   sprintf (sp,"%d",val);
}
/*----------------------------------------------------------------------------
 *        Print a Flash Memory Card Directory
 *---------------------------------------------------------------------------*/
static void cmd_dir (char *par) {
   U32 fsize,files,dirs,i;
   char temp[32],*mask,*next,ch;
   FINFO info;

   mask = get_entry (par, &next);
   if (mask == NULL) {
      mask = "*.*";
   }

   printf ("\nFile System Directory...");
   files = 0;
   dirs  = 0;
   fsize = 0;
   info.fileID  = 0;
   while (ffind (mask,&info) == 0) {
      if (info.attrib & ATTR_DIRECTORY) {
         i = 0;
         while (strlen((const char *)info.name+i) > 41) {
            ch = info.name[i+41];
            info.name[i+41] = 0;
            printf ("\n%-41s", &info.name[i]);
            info.name[i+41] = ch;
            i += 41;
         }
         printf ("\n%-41s    <DIR>       ", &info.name[i]);
         printf ("  %02d.%02d.%04d  %02d:%02d",
                  info.time.day, info.time.mon, info.time.year,
                  info.time.hr, info.time.min);
         dirs++;
      }
      else {
         dot_format (info.size, &temp[0]);
         i = 0;
         while (strlen((const char *)info.name+i) > 41) {
            ch = info.name[i+41];
            info.name[i+41] = 0;
            printf ("\n%-41s", &info.name[i]);
            info.name[i+41] = ch;
            i += 41;
         }
         printf ("\n%-41s %14s ", &info.name[i], temp);
         printf ("  %02d.%02d.%04d  %02d:%02d",
                  info.time.day, info.time.mon, info.time.year,
                  info.time.hr, info.time.min);
         fsize += info.size;
         files++;
      }
   }
   if (info.fileID == 0) {
      printf ("\nNo files...");
   }
   else {
      dot_format (fsize, &temp[0]);
      printf ("\n              %9d File(s)    %21s bytes", files, temp);
   }
}

/*----------------------------------------------------------------------------
 *        Display Command Syntax help
 *---------------------------------------------------------------------------*/
static void cmd_help (char *par) {
   printf (help);
}

/*----------------------------------------------------------------------------
 *        Initialize a Flash Memory Card
 *---------------------------------------------------------------------------*/
static void init_card (void) {
   U32 retv;

   while ((retv = finit ()) != 0) {           /* Wait until the Card is ready*/
      if (retv == 1) {
         printf ("\nSD/MMC Init Failed");
         printf ("\nInsert Memory card and press key...\n");
         getkey ();
      }
      else {
         printf ("\nSD/MMC Card is Unformatted");
		 printf ("\nFormat Memory card then press key...\n");
		 getkey ();
         //strcpy (&in_line[0], "KEIL\r\n");
         //cmd_format (&in_line[0]);
      }
   }
}

/* Run appliication at specified address */
static void cmd_run (char *par)
{
	char *entry, *next;
	int run_addr;
	U32 i;
	PFV fp;
	
	// get run address
	entry = get_entry (par, &next);
	if (entry == NULL) 
		run_addr = DEFAULT_RUN_ADDR;
	else
		run_addr = strtoul(entry);

	fp = (PFV)(run_addr);
	printf("Run applicaiton at 0x%x...\n", run_addr);
	for (i=0x1000;i!=0; i--);
	(*fp)(); 
}

/* Check specified flash blanks are blank or not */
static void cmd_blkchk (char *par)
{
	char *start_index, *end_index, *next;
	U32 start_idx, end_idx;
	U32 IAP_return[2], ret;

	start_index = get_entry (par, &next);
	if (start_index == NULL) 
	{
		printf("\nSector start index missing!\n");
		return;
	}
	start_idx = strtoul(start_index);

	end_index = get_entry (next, &next);  
	if (end_index == NULL)
		end_idx = start_idx;
	else
		end_idx = strtoul(end_index);

	if (start_idx > end_idx)
	{
		printf("\nEnd index should be greater or equal to the start index!\n");
		return;
	}

	ret = IAP_BlankChkSec(start_idx, end_idx, IAP_return);
	if (ret == IAP_STA_CMD_SUCCESS)
	{
		printf("\nSectors from %d to %d are blank.\n", start_idx, end_idx);  	
	}
	else if (ret == IAP_STA_SECTOR_NOT_BLANK )
	{
		printf("\nNot blank.\nOffset of the first non blank word location is 0x%x.\n", IAP_return[1]);
	}
	else
		printf("\nFailed to check blank.\n");

}

/* erase flash sectors with specified start and end index */
static void cmd_erase (char *par)
{
	char *start_index, *end_index, *next;
	U32 start_idx, end_idx;
	U32 ret;

	// get start index
	start_index = get_entry (par, &next);
	if (start_index == NULL) 
	{
		printf("\nSector start index missing!\n");
		return;
	}
	start_idx = strtoul(start_index);

	// get end index
	end_index = get_entry (next, &next);  // Image entry point
	if (end_index == NULL)
		end_idx = start_idx;
	else
		end_idx = strtoul(end_index);
	
	// check the start and end index
	if (start_idx <= bl_sector_endidx)
	{
		printf("\nApplication address overlapped with boot loader!\n");	
		return;
	}
	if (end_idx < start_idx)
	{
		printf("\nEnd index should be greater or equal to the start index!\n");
		return;
	}

	// prepre sectors to erase
	if (IAP_PrepareSec(start_idx, end_idx) != IAP_STA_CMD_SUCCESS)
	{
		printf("\nFailed to prepare sectors.\n");
		return;
	}

	IRQDisable();
	ret = IAP_EraseSec(start_idx, end_idx);
	IRQEnable();
	if (ret == IAP_STA_CMD_SUCCESS)
	{
		printf("\nErased sectors from %d to %d .\n", start_idx, end_idx);
	}
	else
		printf("\nFailed to erase.\n");

}

/* Read Part Identification number */
static void cmd_readid (char *par)
{
	U32 id[1];

	IRQDisable();
	IAP_ReadParID(&id[0]);
	IRQEnable();

	printf("\nPart ID: 0x%x.\n", id[0]);
}

/* progrom a BIN file to a specified address in internal flash */
static void cmd_prog (char *par)
{
	char *fname, *entry, *next;
	FILE *fin;
	U32 app_addr, sector_index, sector_size, sector_size_sum, total, cnt;
	BOOL prog_done, new_sector;
	U32 IAP_return[2];

	// fname: BIN filename, only BIN file is supported now
	fname = get_entry (par, &next);	 // Image file name
	if (fname == NULL) {
	  printf ("\nImage filename missing.\n");
	  return;		
	}
	if (strcmp(fname+strlen(fname)-3, "BIN") != 0 &&
		strcmp(fname+strlen(fname)-3, "bin") != 0)
	{
		printf("\nOnly BIN file is supported.\n");
		return;
	}

	// program address
	entry = get_entry (next, &next);  // Image entry point
	if (entry == NULL)
		app_addr = DEFAULT_PROG_ADDR;
	else
		app_addr = strtoul(entry);	

	// app_addr should be a starting address of some sector,
	// it's better to verify here

	sector_index = getSectorIndex(app_addr);
	if (sector_index == INVALID_RESULT)
	{
		printf("\nInvalid application address!\n");
		return;
	}
	if (sector_index <= bl_sector_endidx)
	{
		printf("\nApplication address overlapped with boot loader!\n");	
		return;
	}


	printf("\nProgramming %s to 0x%x...\n", fname, app_addr);
	
	fin = fopen (fname,"rb");           /* open the file for reading           */
	if (fin == NULL) {
	  printf ("\nFile %s not found!\n",fname);
	  return;
	}

	prog_done = 0;
	new_sector = 1;

	total = 0;
	sector_size_sum = 0;


	do
	{
		cnt = fread (&IAP_Buf, 1, IAP_BUF_SIZE, fin);
		if (cnt == 0) /* error of EOF */
		{
			prog_done = 1;
			break;
		}
		else
		{
			if (new_sector)
			{
				if (IAP_PrepareErase(sector_index) != 0 )
				{
					printf("\nFailed to prepare/erase sector %d.\n", sector_index);
					prog_done = 1;
					break;
				}
#if 0				
				// prepre sector [sector_index] to erase
				if(IAP_PrepareSec(sector_index, sector_index) != IAP_STA_CMD_SUCCESS)
				{
					printf("\nFailed to prepare sector %d.\n", sector_index);
					prog_done = 1;
					break;
				}
				// erase sector [sector_index]
				IRQDisable();
				if (IAP_EraseSec(sector_index, sector_index) != IAP_STA_CMD_SUCCESS)
				{
					printf("\nFailed to erase sector %d.\n", sector_index);	
					prog_done = 1;
					break;
				}
				IRQEnable();
#endif

				sector_size = (getSectorSize(sector_index) << 10);
				sector_size_sum = 0;
				new_sector = 0;
			}
			
			if (IAP_Program(sector_index, app_addr) != 0)
			{
				printf("\nFailed to program at 0x%x.\n", app_addr);
				prog_done = 1;
				break;
			}

#if 0			
			// program 1kb [app_addr]			
			// prepre sector [sector_index] to write
			if(IAP_PrepareSec(sector_index, sector_index) != IAP_STA_CMD_SUCCESS)
			{
				printf("\nFailed to prepare sector %d.\n", sector_index);
				prog_done = 1;
				break;
			}
			IRQDisable();
			if ((IAP_CopyRAMToFlash(app_addr, (U32)IAP_Buf, IAP_BUF_SIZE)) != IAP_STA_CMD_SUCCESS)
			{
				printf("\nFailed to program at 0x%x.\n", app_addr);
				prog_done = 1;
				break;
			}
			if (IAP_Compare(app_addr, (U32)IAP_Buf, IAP_BUF_SIZE, IAP_return) != IAP_STA_CMD_SUCCESS)
			{
				printf("\nVerify failed at 0x%x.\n", app_addr);
				prog_done = 1;
				break;
			}
			IRQEnable();
#endif

			app_addr += IAP_BUF_SIZE;
			total += cnt;
			sector_size_sum += 	IAP_BUF_SIZE;
			if (sector_size_sum	== sector_size)
			{
				sector_index++;
				new_sector = 1;
			}
		}
				
	} while (prog_done==0);
   
   printf("\n%d bytes programmed.\n", total);

   fclose (fin); 		
}

/* prepare and erase sector with specified index,
	if OK, return 0, otherwise return the error code */
static int IAP_PrepareErase(U32 sector_index)
{
	int ret = 0;

	// prepre sector [sector_index] to erase
	if(IAP_PrepareSec(sector_index, sector_index) != IAP_STA_CMD_SUCCESS)
	{
		return 10;
	}
	// erase sector [sector_index]
	IRQDisable();
	if (IAP_EraseSec(sector_index, sector_index) != IAP_STA_CMD_SUCCESS)
	{
		ret = 11;
	}
	IRQEnable();
	
	return ret;	
}

/* program content in IAP_Buf to internal flash with address of 
	app_addr, and sector index of sector_index.
	if ok, return 0, otherwise return the error code. */
static int IAP_Program(U32 sector_index, U32 app_addr)
{
	U32 IAP_return[2];

	// program 1kb [app_addr]			
	// prepre sector [sector_index] to write
	if(IAP_PrepareSec(sector_index, sector_index) != IAP_STA_CMD_SUCCESS)
	{
		return 10;
	}

	IRQDisable();
	if ((IAP_CopyRAMToFlash(app_addr, (U32)IAP_Buf, IAP_BUF_SIZE)) != IAP_STA_CMD_SUCCESS)
	{
		IRQEnable();
		return 12;
	}
	IRQEnable();

	if (IAP_Compare(app_addr, (U32)IAP_Buf, IAP_BUF_SIZE, IAP_return) != IAP_STA_CMD_SUCCESS)
	{
		return 13;
	}

	return 0;	
}

/* Display info of bootloader */
static void cmd_bootinfo (char *par)
{
	printf("  Bootloader:\n");
	printf("             Entry:    0x%x\n", (U32)Image$$ER_IROM1$$RO$$Base);
	printf("             Size:     0x%x bytes (%d)\n", bl_size, bl_size);
	printf("             Sectors:  %d to %d\n", bl_sector_startidx, bl_sector_endidx);
	printf("             Version:  %s\n", __DATE__);
}

static void init_bootpara(void)
{
	bl_size = (U32)Image$$ER_IROM1$$RO$$Length + (U32)Image$$RW_IRAM1$$RW$$Length;
	bl_sector_startidx = getSectorIndex((U32)Image$$ER_IROM1$$RO$$Base);
	bl_sector_endidx = getEndSectorIndex(bl_size, bl_sector_startidx);
}

/* string -> signed int*/
static int strtoul(char *s)
{
	int ret;
	int radix = 10;
	int negative = 0;
	int i;

	ret = 0;
	if(*s == '-') 
	{
		negative = 1;
		s++;
	}
	else if(*s == '0')
	{
		s++;
		if(*s == 'x')
		{
			s++;
			radix = 0x10;
		}
	}

	while (*s) {
		if (*s >= '0' && *s <= '9')
			i = *s - '0';
		else if (*s >= 'a' && *s <= 'f')
			i = *s - 'a' + 0xa;
		else if (*s >= 'A' && *s <= 'F')
			i = *s - 'A' + 0xa;
		else
			break;
		if(i >= radix) break;
		ret = (ret * radix) + i;
		s++;
	}

	return negative?(-ret):ret;
}

/*----------------------------------------------------------------------------
 *        Main: 
 *---------------------------------------------------------------------------*/
int main (void) {
   char *sp,*cp,*next;
   U32 i;

   init_bootpara();
   init_comm ();                              /* init communication interface*/
   printf (intro);                            /* display example info        */
   printf (help);

   init_card ();
   while (1) {
      printf ("\nCmd> ");                     /* display prompt              */
      fflush (stdout);

	  /* end with ENTER */
                                              /* get command line input      */
      if (getline (in_line, sizeof (in_line)) == __FALSE) {
         continue;
      }

      sp = get_entry (&in_line[0], &next);
      if (*sp == 0) {
         continue;
      }

      for (cp = sp; *cp && *cp != ' '; cp++) {
         *cp = toupper (*cp);                 /* command to upper-case       */
      }
      for (i = 0; i < CMD_COUNT; i++) {
         if (strcmp (sp, (const char *)&cmd[i].val)) {
            continue;
         }
         init_card();                         /* check if card is removed    */
         cmd[i].func (next);                  /* execute command function    */
         break;
      }
      if (i == CMD_COUNT) {
        printf ("\nCommand error\n");
      }
   }
}


/*----------------------------------------------------------------------------
 * end of file
 *---------------------------------------------------------------------------*/
